/**
 * WEBFUL Analytics - Script de tracking ultra-léger
 * Version: 1.0.0
 * Taille minifiée: < 1ko
 */

(function() {
    'use strict';

    // Marqueur de version pour détection de mise à jour
    window.WEBFUL_PLUGIN_VERSION = '2.0.0';

    // Configuration injectée par WordPress
    var config = window.webfulConfig || {};
    var siteId = config.siteId;
    var apiUrl = config.apiUrl;

    // Vérifier que la configuration est présente
    if (!siteId || !apiUrl) {
        console.warn('WEBFUL: Configuration manquante');
        return;
    }

    // Variables de tracking
    var startTime = Date.now();
    var sessionId = getOrCreateSessionId();
    var sent = false;

    /**
     * Générer ou récupérer l'ID de session
     */
    function getOrCreateSessionId() {
        var key = 'webful_sid';
        var sid = sessionStorage.getItem(key);

        if (!sid) {
            sid = generateId();
            sessionStorage.setItem(key, sid);
        }

        return sid;
    }

    /**
     * Générer un ID unique au format WBF-timestamp-random
     */
    function generateId() {
        var timestamp = Date.now(); // 13 chiffres
        var random = Math.random().toString(36).substr(2, 9); // 9 caractères
        return 'WBF-' + timestamp + '-' + random;
    }

    /**
     * Obtenir le hash de l'IP (simulé côté client avec un ID unique par navigateur)
     */
    function getVisitorId() {
        var key = 'webful_vid';
        var vid = localStorage.getItem(key);

        if (!vid) {
            vid = generateId();
            localStorage.setItem(key, vid);
        }

        return vid;
    }

    /**
     * Calculer le temps passé sur la page
     */
    function getTimeSpent() {
        return Math.round((Date.now() - startTime) / 1000);
    }

    /**
     * Envoyer les données de tracking
     */
    function sendTracking() {
        if (sent) return; // Ne pas envoyer deux fois
        sent = true;

        var data = {
            site_id: siteId,
            session_id: sessionId,
            ip_hash: getVisitorId(),
            url_page: window.location.href,
            referrer: document.referrer || null,
            user_agent: navigator.userAgent,
            temps_passe: getTimeSpent()
        };

        // Utiliser sendBeacon si disponible (plus fiable pour les événements de sortie)
        if (navigator.sendBeacon) {
            var blob = new Blob([JSON.stringify(data)], { type: 'application/json' });
            navigator.sendBeacon(apiUrl, blob);
        } else {
            // Fallback avec fetch
            fetch(apiUrl, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(data),
                keepalive: true
            }).catch(function() {
                // Ignorer les erreurs silencieusement
            });
        }
    }

    /**
     * Envoyer un événement de conversion
     */
    function sendConversion(type, label) {
        var eventApiUrl = apiUrl.replace('/track.php', '/track-event.php');

        var data = {
            site_id: siteId,
            session_id: sessionId,
            visitor_id: getVisitorId(),
            conversion_type: type,
            conversion_label: label || null,
            page_url: window.location.href,
            referrer: document.referrer || null,
            user_agent: navigator.userAgent
        };

        // Envoyer avec fetch
        if (navigator.sendBeacon) {
            var blob = new Blob([JSON.stringify(data)], { type: 'application/json' });
            navigator.sendBeacon(eventApiUrl, blob);
        } else {
            fetch(eventApiUrl, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(data)
            }).catch(function() {
                // Ignorer les erreurs silencieusement
            });
        }
    }

    /**
     * Détecter les conversions automatiques
     */
    function setupConversionTracking() {
        // 1. Détecter les clics sur tel: et mailto:
        document.addEventListener('click', function(e) {
            var target = e.target;

            // Remonter jusqu'au lien parent si on clique sur un élément enfant
            while (target && target.tagName !== 'A') {
                target = target.parentElement;
            }

            if (!target || target.tagName !== 'A') return;

            var href = target.getAttribute('href');
            if (!href) return;

            // Clic sur numéro de téléphone
            if (href.indexOf('tel:') === 0) {
                var phone = href.replace('tel:', '').replace(/\s/g, '');
                sendConversion('tel_click', phone);
            }

            // Clic sur adresse email
            if (href.indexOf('mailto:') === 0) {
                var email = href.replace('mailto:', '').split('?')[0];
                sendConversion('email_click', email);
            }
        });

        // 2. Détecter les soumissions de formulaire
        document.addEventListener('submit', function(e) {
            var form = e.target;
            if (form.tagName === 'FORM') {
                var formId = form.id || form.name || form.action || 'formulaire';
                sendConversion('form_submit', formId);
            }
        });

        // 3. Détecter les visites de pages importantes
        var path = window.location.pathname.toLowerCase();
        var conversionPages = ['/contact', '/merci', '/thank-you', '/confirmation', '/success', '/remerciement'];

        for (var i = 0; i < conversionPages.length; i++) {
            if (path.indexOf(conversionPages[i]) !== -1) {
                sendConversion('page_visit', path);
                break;
            }
        }
    }

    /**
     * Envoyer les données lors de la sortie de la page
     */
    function setupBeforeUnload() {
        // Événements de sortie
        window.addEventListener('beforeunload', sendTracking);
        window.addEventListener('pagehide', sendTracking);

        // Visibilité de la page (changement d'onglet)
        document.addEventListener('visibilitychange', function() {
            if (document.visibilityState === 'hidden') {
                sendTracking();
            }
        });
    }

    /**
     * Initialiser le tracking
     */
    function init() {
        // Attendre que le DOM soit chargé
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', function() {
                setupBeforeUnload();
                setupConversionTracking();
            });
        } else {
            setupBeforeUnload();
            setupConversionTracking();
        }

        // Envoyer aussi après un certain temps (fallback)
        setTimeout(function() {
            if (!sent && getTimeSpent() > 5) {
                sendTracking();
            }
        }, 30000); // Après 30 secondes
    }

    // Lancer le tracking
    init();
})();
