<?php
/**
 * Classe de gestion de l'interface d'administration WEBFUL
 *
 * @package WEBFUL
 * @since 1.0.0
 */

// Empêcher l'accès direct
if (!defined('ABSPATH')) {
    exit;
}

class WEBFUL_Admin {

    /**
     * Instance de l'API
     */
    private $api;

    /**
     * Constructeur
     */
    public function __construct() {
        $this->api = new WEBFUL_API();
        $this->init_hooks();
    }

    /**
     * Initialiser les hooks WordPress
     */
    private function init_hooks() {
        // Ajouter la page de menu
        add_action('admin_menu', array($this, 'add_admin_menu'));

        // Enregistrer les scripts et styles admin
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));

        // Afficher la notice de consentement RGPD
        add_action('admin_notices', array($this, 'show_gdpr_consent_notice'));

        // Traiter les actions AJAX
        add_action('wp_ajax_webful_connect', array($this, 'ajax_connect'));
        add_action('wp_ajax_webful_disconnect', array($this, 'ajax_disconnect'));
        add_action('wp_ajax_webful_save_settings', array($this, 'ajax_save_settings'));
        add_action('wp_ajax_webful_check_quota', array($this, 'ajax_check_quota'));
        add_action('wp_ajax_webful_accept_consent', array($this, 'ajax_accept_consent'));
        add_action('wp_ajax_webful_decline_consent', array($this, 'ajax_decline_consent'));
    }

    /**
     * Ajouter le menu d'administration
     */
    public function add_admin_menu() {
        add_menu_page(
            __('WEBFUL Analytics', 'webful'),
            __('WEBFUL', 'webful'),
            'manage_options',
            'webful',
            array($this, 'render_admin_page'),
            'dashicons-chart-line',
            30
        );
    }

    /**
     * Charger les assets admin
     */
    public function enqueue_admin_assets($hook) {
        // Charger uniquement sur notre page
        if ($hook !== 'toplevel_page_webful') {
            return;
        }

        wp_enqueue_style(
            'webful-admin',
            WEBFUL_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            WEBFUL_VERSION
        );

        wp_enqueue_script(
            'webful-admin',
            WEBFUL_PLUGIN_URL . 'assets/js/admin.js',
            array('jquery'),
            WEBFUL_VERSION,
            true
        );

        // Passer les données à JavaScript
        wp_localize_script('webful-admin', 'webfulAdmin', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('webful_admin'),
            'i18n' => array(
                'connecting' => __('Connexion...', 'webful'),
                'disconnecting' => __('Déconnexion...', 'webful'),
                'saving' => __('Enregistrement...', 'webful'),
                'checking' => __('Vérification...', 'webful'),
                'success' => __('Succès', 'webful'),
                'error' => __('Erreur', 'webful')
            )
        ));
    }

    /**
     * Afficher la page d'administration
     */
    public function render_admin_page() {
        $is_connected = $this->api->is_connected();
        $site_info = $this->api->get_site_info();

        ?>
        <div class="wrap webful-admin">
            <h1><?php _e('WEBFUL Analytics', 'webful'); ?></h1>

            <div class="webful-container">

                <?php if ($is_connected): ?>
                    <!-- Site connecté -->
                    <?php $this->render_connected_view($site_info); ?>
                <?php else: ?>
                    <!-- Site non connecté -->
                    <?php $this->render_connection_form(); ?>
                <?php endif; ?>

            </div>
        </div>
        <?php
    }

    /**
     * Afficher le formulaire de connexion
     */
    private function render_connection_form() {
        ?>
        <div class="webful-card">
            <div class="webful-card-header">
                <h2><?php _e('Connecter votre site à WEBFUL', 'webful'); ?></h2>
            </div>
            <div class="webful-card-body">
                <p class="description">
                    <?php _e('Entrez vos identifiants WEBFUL pour connecter votre site WordPress et commencer à suivre vos statistiques.', 'webful'); ?>
                </p>

                <form id="webful-connect-form" class="webful-form">
                    <div class="form-group">
                        <label for="webful_email"><?php _e('Email', 'webful'); ?></label>
                        <input
                            type="email"
                            id="webful_email"
                            name="email"
                            class="regular-text"
                            required
                            placeholder="<?php esc_attr_e('votre@email.com', 'webful'); ?>"
                        >
                    </div>

                    <div class="form-group">
                        <label for="webful_password"><?php _e('Mot de passe', 'webful'); ?></label>
                        <input
                            type="password"
                            id="webful_password"
                            name="password"
                            class="regular-text"
                            required
                        >
                    </div>

                    <div id="webful-message" class="webful-message" style="display: none;"></div>

                    <div class="form-actions">
                        <button type="submit" class="button button-primary button-large">
                            <?php _e('Connecter mon site', 'webful'); ?>
                        </button>
                    </div>
                </form>

                <div class="webful-help">
                    <p>
                        <strong><?php _e('Vous n\'avez pas encore de compte WEBFUL ?', 'webful'); ?></strong><br>
                        <a href="<?php echo esc_url(str_replace('/api', '/inscription.php', WEBFUL_API_BASE)); ?>" target="_blank">
                            <?php _e('Créer un compte gratuitement', 'webful'); ?>
                        </a>
                    </p>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Afficher la vue connectée
     */
    private function render_connected_view($site_info) {
        $quota = $this->api->check_quota();
        $quota_error = is_wp_error($quota);

        ?>
        <div class="webful-grid">

            <!-- Statut de connexion -->
            <div class="webful-card">
                <div class="webful-card-header">
                    <h2><?php _e('Statut de la connexion', 'webful'); ?></h2>
                </div>
                <div class="webful-card-body">
                    <div class="webful-status webful-status-connected">
                        <span class="dashicons dashicons-yes-alt"></span>
                        <span><?php _e('Connecté', 'webful'); ?></span>
                    </div>

                    <table class="webful-info-table">
                        <tr>
                            <td><strong><?php _e('Site ID:', 'webful'); ?></strong></td>
                            <td><code><?php echo esc_html($site_info['site_id']); ?></code></td>
                        </tr>
                        <tr>
                            <td><strong><?php _e('URL:', 'webful'); ?></strong></td>
                            <td><?php echo esc_html(get_site_url()); ?></td>
                        </tr>
                        <tr>
                            <td><strong><?php _e('Compte:', 'webful'); ?></strong></td>
                            <td><?php echo esc_html($site_info['user_email']); ?></td>
                        </tr>
                    </table>

                    <div class="form-actions">
                        <a href="<?php echo esc_url($this->api->get_dashboard_url()); ?>" class="button button-primary" target="_blank">
                            <span class="dashicons dashicons-chart-line"></span>
                            <?php _e('Voir mes statistiques', 'webful'); ?>
                        </a>
                        <button type="button" id="webful-disconnect-btn" class="button button-secondary">
                            <?php _e('Déconnecter', 'webful'); ?>
                        </button>
                    </div>
                </div>
            </div>

            <!-- Quota -->
            <div class="webful-card">
                <div class="webful-card-header">
                    <h2><?php _e('Quota de vues', 'webful'); ?></h2>
                </div>
                <div class="webful-card-body">
                    <?php if ($quota_error): ?>
                        <div class="webful-notice webful-notice-error">
                            <p><?php echo esc_html($quota->get_error_message()); ?></p>
                        </div>
                    <?php else: ?>
                        <div class="webful-quota">
                            <div class="webful-quota-bar">
                                <?php
                                $used = $quota['vues_utilisees'];
                                $total = $quota['vues_totales'];
                                $percentage = $total > 0 ? min(100, ($used / $total) * 100) : 0;
                                $percentage_class = $percentage >= 90 ? 'danger' : ($percentage >= 75 ? 'warning' : 'success');
                                ?>
                                <div class="webful-quota-progress">
                                    <div class="webful-quota-fill webful-quota-<?php echo $percentage_class; ?>" style="width: <?php echo $percentage; ?>%"></div>
                                </div>
                                <div class="webful-quota-text">
                                    <?php
                                    printf(
                                        __('%s / %s vues utilisées ce mois (%s%%)', 'webful'),
                                        number_format($used, 0, ',', ' '),
                                        $total === 999999999 ? __('illimité', 'webful') : number_format($total, 0, ',', ' '),
                                        number_format($percentage, 1, ',', ' ')
                                    );
                                    ?>
                                </div>
                            </div>

                            <?php if (!$quota['quota_ok']): ?>
                                <div class="webful-notice webful-notice-warning">
                                    <p><strong><?php _e('Quota atteint !', 'webful'); ?></strong></p>
                                    <p><?php _e('Le tracking est actuellement désactivé. Mettez à niveau votre plan pour continuer à collecter des données.', 'webful'); ?></p>
                                </div>
                            <?php endif; ?>

                            <p class="description">
                                <strong><?php _e('Plan actuel:', 'webful'); ?></strong>
                                <?php echo esc_html(ucfirst($quota['plan'])); ?>
                            </p>

                            <button type="button" id="webful-refresh-quota" class="button button-secondary">
                                <span class="dashicons dashicons-update"></span>
                                <?php _e('Actualiser', 'webful'); ?>
                            </button>
                        </div>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Paramètres -->
            <div class="webful-card webful-card-full">
                <div class="webful-card-header">
                    <h2><?php _e('Paramètres', 'webful'); ?></h2>
                </div>
                <div class="webful-card-body">
                    <form id="webful-settings-form" class="webful-form">

                        <div class="form-group">
                            <label>
                                <input
                                    type="checkbox"
                                    name="tracking_enabled"
                                    id="webful_tracking_enabled"
                                    value="1"
                                    <?php checked($site_info['tracking_enabled'], '1'); ?>
                                >
                                <?php _e('Activer le tracking sur ce site', 'webful'); ?>
                            </label>
                            <p class="description">
                                <?php _e('Désactivez cette option pour arrêter temporairement le suivi sans déconnecter le site.', 'webful'); ?>
                            </p>
                        </div>

                        <div class="form-group">
                            <label>
                                <input
                                    type="checkbox"
                                    name="check_quota"
                                    id="webful_check_quota"
                                    value="1"
                                    <?php checked($site_info['check_quota'], '1'); ?>
                                >
                                <?php _e('Vérifier le quota avant chaque tracking', 'webful'); ?>
                            </label>
                            <p class="description">
                                <?php _e('Recommandé : vérifie automatiquement si votre quota est atteint avant d\'envoyer les données.', 'webful'); ?>
                            </p>
                        </div>

                        <div class="form-group">
                            <label>
                                <input
                                    type="checkbox"
                                    name="exclude_admins"
                                    id="webful_exclude_admins"
                                    value="1"
                                    <?php checked($site_info['exclude_admins'], '1'); ?>
                                >
                                <?php _e('Ne pas tracker les administrateurs connectés', 'webful'); ?>
                            </label>
                            <p class="description">
                                <?php _e('Activé par défaut : vos propres visites ne seront pas comptabilisées dans les statistiques.', 'webful'); ?>
                            </p>
                        </div>

                        <div class="form-group">
                            <label for="webful_excluded_pages">
                                <?php _e('Pages à exclure du tracking', 'webful'); ?>
                            </label>
                            <textarea
                                id="webful_excluded_pages"
                                name="excluded_pages"
                                rows="5"
                                class="large-text"
                                placeholder="<?php esc_attr_e('/contact&#10;/merci&#10;mon-slug-de-page', 'webful'); ?>"
                            ><?php echo esc_textarea($site_info['excluded_pages']); ?></textarea>
                            <p class="description">
                                <?php _e('Entrez une URL, un chemin ou un slug par ligne. Exemples : /contact, /merci, mon-article', 'webful'); ?>
                            </p>
                        </div>

                        <div id="webful-settings-message" class="webful-message" style="display: none;"></div>

                        <div class="form-actions">
                            <button type="submit" class="button button-primary">
                                <?php _e('Enregistrer les paramètres', 'webful'); ?>
                            </button>
                        </div>

                    </form>
                </div>
            </div>

        </div>
        <?php
    }

    /**
     * AJAX: Connecter le site
     */
    public function ajax_connect() {
        check_ajax_referer('webful_admin', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission refusée', 'webful')));
        }

        $email = isset($_POST['email']) ? sanitize_email($_POST['email']) : '';
        $password = isset($_POST['password']) ? $_POST['password'] : '';

        if (empty($email) || empty($password)) {
            wp_send_json_error(array('message' => __('Email et mot de passe requis', 'webful')));
        }

        $result = $this->api->register_site($email, $password);

        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
        }

        // Vider le cache WordPress pour forcer le rechargement du script avec le nouveau site_id
        $this->clear_wordpress_cache();

        wp_send_json_success(array('message' => __('Site connecté avec succès !', 'webful')));
    }

    /**
     * AJAX: Déconnecter le site
     */
    public function ajax_disconnect() {
        check_ajax_referer('webful_admin', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission refusée', 'webful')));
        }

        $this->api->disconnect();

        wp_send_json_success(array('message' => __('Site déconnecté avec succès', 'webful')));
    }

    /**
     * AJAX: Enregistrer les paramètres
     */
    public function ajax_save_settings() {
        check_ajax_referer('webful_admin', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission refusée', 'webful')));
        }

        $tracking_enabled = isset($_POST['tracking_enabled']) && $_POST['tracking_enabled'] === '1' ? '1' : '0';
        $check_quota = isset($_POST['check_quota']) && $_POST['check_quota'] === '1' ? '1' : '0';
        $exclude_admins = isset($_POST['exclude_admins']) && $_POST['exclude_admins'] === '1' ? '1' : '0';
        $excluded_pages = isset($_POST['excluded_pages']) ? sanitize_textarea_field($_POST['excluded_pages']) : '';

        update_option('webful_tracking_enabled', $tracking_enabled);
        update_option('webful_check_quota', $check_quota);
        update_option('webful_exclude_admins', $exclude_admins);
        update_option('webful_excluded_pages', $excluded_pages);

        // Rafraîchir le cache du quota
        delete_transient('webful_quota_check');

        wp_send_json_success(array('message' => __('Paramètres enregistrés avec succès', 'webful')));
    }

    /**
     * AJAX: Vérifier le quota
     */
    public function ajax_check_quota() {
        check_ajax_referer('webful_admin', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission refusée', 'webful')));
        }

        // Rafraîchir le cache
        delete_transient('webful_quota_check');

        $quota = $this->api->check_quota();

        if (is_wp_error($quota)) {
            wp_send_json_error(array('message' => $quota->get_error_message()));
        }

        wp_send_json_success(array(
            'message' => __('Quota actualisé', 'webful'),
            'quota' => $quota
        ));
    }

    /**
     * Vider le cache WordPress (compatibilité avec les plugins de cache populaires)
     * Appelé après connexion/reconnexion pour forcer le rechargement du script avec le nouveau site_id
     */
    private function clear_wordpress_cache() {
        // WP Rocket
        if (function_exists('rocket_clean_domain')) {
            rocket_clean_domain();
        }

        // W3 Total Cache
        if (function_exists('w3tc_flush_all')) {
            w3tc_flush_all();
        }

        // WP Super Cache
        if (function_exists('wp_cache_clear_cache')) {
            wp_cache_clear_cache();
        }

        // Autoptimize
        if (class_exists('autoptimizeCache')) {
            autoptimizeCache::clearall();
        }

        // LiteSpeed Cache
        if (class_exists('LiteSpeed_Cache_API')) {
            LiteSpeed_Cache_API::purge_all();
        }

        // Cache natif WordPress (object cache)
        wp_cache_flush();

        // Supprimer les transients WEBFUL
        delete_transient('webful_quota_check');
    }

    /**
     * Afficher la notice de consentement RGPD
     */
    public function show_gdpr_consent_notice() {
        // Vérifier si on doit afficher la notice
        $show_notice = get_option('webful_show_consent_notice', '0');
        $gdpr_consent = get_option('webful_gdpr_consent', '0');

        // Ne pas afficher si déjà consenti ou refusé
        if ($show_notice === '0' || $gdpr_consent !== '0') {
            return;
        }

        // Afficher la notice uniquement sur les pages d'administration
        $screen = get_current_screen();
        ?>
        <div class="notice notice-info is-dismissible webful-gdpr-notice" style="position: relative; padding: 20px;">
            <div style="display: flex; align-items: start; gap: 15px;">
                <div style="flex-shrink: 0;">
                    <svg width="48" height="48" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <path d="M12 2L2 7L12 12L22 7L12 2Z" fill="#2271b1" opacity="0.2"/>
                        <path d="M2 17L12 22L22 17M2 12L12 17L22 12" stroke="#2271b1" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                    </svg>
                </div>
                <div style="flex: 1;">
                    <h3 style="margin-top: 0; margin-bottom: 10px; font-size: 16px;">
                        🔒 WEBFUL Analytics - Consentement pour le traitement des données
                    </h3>
                    <p style="margin-bottom: 15px;">
                        <strong>WEBFUL Analytics</strong> est un outil d'analyse de trafic qui vous aide à comprendre vos visiteurs.
                        Pour fonctionner, ce plugin envoie des données anonymisées de navigation vers les serveurs WEBFUL (webful.fr).
                    </p>

                    <div style="background: #f0f6fc; border-left: 4px solid #2271b1; padding: 12px; margin-bottom: 15px;">
                        <p style="margin: 0; font-weight: 600; margin-bottom: 8px;">📊 Données collectées :</p>
                        <ul style="margin: 0; padding-left: 20px;">
                            <li>Pages visitées (URLs)</li>
                            <li>Temps passé sur chaque page</li>
                            <li>Source de trafic (referrer)</li>
                            <li>Adresse IP <strong>hachée de manière irréversible</strong> (anonymisation totale)</li>
                        </ul>
                    </div>

                    <div style="background: #f0fdf4; border-left: 4px solid #22c55e; padding: 12px; margin-bottom: 15px;">
                        <p style="margin: 0; font-weight: 600; margin-bottom: 8px;">✅ Protection de la vie privée :</p>
                        <ul style="margin: 0; padding-left: 20px;">
                            <li><strong>Aucun cookie</strong> n'est utilisé</li>
                            <li><strong>Aucune donnée personnelle identifiable</strong> n'est collectée</li>
                            <li>Les adresses IP sont hachées (SHA-256) de manière <strong>irréversible</strong></li>
                            <li>Conformité <strong>RGPD</strong> native</li>
                            <li>Données stockées en <strong>France</strong></li>
                        </ul>
                    </div>

                    <p style="margin-bottom: 15px;">
                        En acceptant, vous autorisez WEBFUL Analytics à envoyer ces données anonymisées vers webful.fr pour générer vos statistiques.
                        Vous pouvez retirer votre consentement à tout moment depuis les réglages du plugin.
                    </p>

                    <p style="font-size: 12px; color: #666; margin-bottom: 15px;">
                        Pour plus d'informations :
                        <a href="https://webful.fr/politique-confidentialite" target="_blank">Politique de confidentialité</a> |
                        <a href="https://webful.fr/documentation" target="_blank">Documentation</a>
                    </p>

                    <div style="display: flex; gap: 10px; flex-wrap: wrap;">
                        <button type="button" class="button button-primary button-large" id="webful-accept-consent">
                            ✅ J'accepte et active le tracking
                        </button>
                        <button type="button" class="button button-secondary button-large" id="webful-decline-consent">
                            ❌ Je refuse
                        </button>
                    </div>
                </div>
            </div>
        </div>

        <script type="text/javascript">
        jQuery(document).ready(function($) {
            // Accepter le consentement
            $('#webful-accept-consent').on('click', function() {
                var $button = $(this);
                $button.prop('disabled', true).text('Activation en cours...');

                $.post(ajaxurl, {
                    action: 'webful_accept_consent',
                    _ajax_nonce: '<?php echo wp_create_nonce('webful_consent'); ?>'
                }, function(response) {
                    if (response.success) {
                        $('.webful-gdpr-notice').fadeOut(function() {
                            $(this).remove();
                        });
                        // Afficher un message de succès
                        $('<div class="notice notice-success is-dismissible"><p><strong>WEBFUL Analytics activé !</strong> Vous pouvez maintenant connecter votre site depuis le menu WEBFUL.</p></div>')
                            .insertAfter('.wp-header-end')
                            .delay(5000)
                            .fadeOut();
                    } else {
                        alert('Erreur : ' + (response.data || 'Impossible de sauvegarder le consentement'));
                        $button.prop('disabled', false).text('✅ J\'accepte et active le tracking');
                    }
                });
            });

            // Refuser le consentement
            $('#webful-decline-consent').on('click', function() {
                if (!confirm('Êtes-vous sûr de vouloir refuser ? Le plugin ne pourra pas fonctionner sans votre consentement.')) {
                    return;
                }

                var $button = $(this);
                $button.prop('disabled', true).text('Enregistrement...');

                $.post(ajaxurl, {
                    action: 'webful_decline_consent',
                    _ajax_nonce: '<?php echo wp_create_nonce('webful_consent'); ?>'
                }, function(response) {
                    if (response.success) {
                        $('.webful-gdpr-notice').fadeOut(function() {
                            $(this).remove();
                        });
                        // Afficher un message d'information
                        $('<div class="notice notice-warning is-dismissible"><p><strong>Consentement refusé.</strong> Le plugin WEBFUL Analytics ne sera pas activé. Vous pouvez le désactiver complètement si vous le souhaitez.</p></div>')
                            .insertAfter('.wp-header-end')
                            .delay(8000)
                            .fadeOut();
                    } else {
                        alert('Erreur : ' + (response.data || 'Impossible de sauvegarder votre choix'));
                        $button.prop('disabled', false).text('❌ Je refuse');
                    }
                });
            });
        });
        </script>

        <style>
        .webful-gdpr-notice {
            border-left-color: #2271b1 !important;
        }
        .webful-gdpr-notice h3 {
            color: #1d2327;
        }
        .webful-gdpr-notice .button-large {
            padding: 8px 24px;
            height: auto;
            font-size: 14px;
        }
        </style>
        <?php
    }

    /**
     * AJAX : Accepter le consentement RGPD
     */
    public function ajax_accept_consent() {
        check_ajax_referer('webful_consent');

        update_option('webful_gdpr_consent', '1');
        update_option('webful_gdpr_consent_date', current_time('mysql'));
        update_option('webful_show_consent_notice', '0');
        update_option('webful_tracking_enabled', '1'); // Activer le tracking

        wp_send_json_success('Consentement enregistré');
    }

    /**
     * AJAX : Refuser le consentement RGPD
     */
    public function ajax_decline_consent() {
        check_ajax_referer('webful_consent');

        update_option('webful_gdpr_consent', '-1'); // -1 = refusé explicitement
        update_option('webful_gdpr_consent_date', current_time('mysql'));
        update_option('webful_show_consent_notice', '0');
        update_option('webful_tracking_enabled', '0'); // Désactiver le tracking

        wp_send_json_success('Refus enregistré');
    }
}
