<?php
/**
 * Classe de communication avec l'API WEBFUL
 *
 * @package WEBFUL
 * @since 1.0.0
 */

// Empêcher l'accès direct
if (!defined('ABSPATH')) {
    exit;
}

class WEBFUL_API {

    /**
     * URL de base de l'API
     */
    private $api_base;

    /**
     * Constructeur
     */
    public function __construct() {
        $this->api_base = WEBFUL_API_BASE;
    }

    /**
     * Enregistrer un nouveau site auprès de WEBFUL
     *
     * @param string $email Email de l'utilisateur WEBFUL
     * @param string $password Mot de passe de l'utilisateur WEBFUL
     * @return array|WP_Error Résultat de l'enregistrement ou erreur
     */
    public function register_site($email, $password) {
        $site_url = get_site_url();
        $site_name = get_bloginfo('name');

        $response = wp_remote_post($this->api_base . '/register-site.php', array(
            'timeout' => 30,
            'headers' => array('Content-Type' => 'application/json'),
            'body' => json_encode(array(
                'email' => $email,
                'password' => $password,
                'site_url' => $site_url,
                'site_name' => $site_name
            ))
        ));

        if (is_wp_error($response)) {
            return $response;
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (!$data || !isset($data['success'])) {
            return new WP_Error('invalid_response', __('Réponse invalide du serveur', 'webful'));
        }

        if (!$data['success']) {
            $error_message = isset($data['error']) ? $data['error'] : __('Erreur inconnue', 'webful');
            return new WP_Error('registration_failed', $error_message);
        }

        // Enregistrer les informations du site
        if (isset($data['data']['site_id'])) {
            update_option('webful_site_id', $data['data']['site_id']);
        }

        if (isset($data['data']['api_key'])) {
            update_option('webful_api_key', $data['data']['api_key']);
        }

        if (isset($data['data']['site_url'])) {
            update_option('webful_site_url', $data['data']['site_url']);
        }

        // Enregistrer l'email pour référence
        update_option('webful_user_email', $email);

        return $data;
    }

    /**
     * Vérifier le quota de vues disponibles
     *
     * @return array|WP_Error Informations sur le quota ou erreur
     */
    public function check_quota() {
        $site_id = get_option('webful_site_id');
        $api_key = get_option('webful_api_key');

        if (empty($site_id) || empty($api_key)) {
            return new WP_Error('not_configured', __('Site non configuré', 'webful'));
        }

        $response = wp_remote_post($this->api_base . '/check-quota.php', array(
            'timeout' => 15,
            'headers' => array('Content-Type' => 'application/json'),
            'body' => json_encode(array(
                'site_id' => $site_id,
                'api_key' => $api_key
            ))
        ));

        if (is_wp_error($response)) {
            return $response;
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (!$data || !isset($data['success'])) {
            return new WP_Error('invalid_response', __('Réponse invalide du serveur', 'webful'));
        }

        if (!$data['success']) {
            $error_message = isset($data['error']) ? $data['error'] : __('Erreur inconnue', 'webful');
            return new WP_Error('quota_check_failed', $error_message);
        }

        return $data['data'];
    }

    /**
     * Vérifier si le site est connecté à WEBFUL
     *
     * @return bool True si connecté, false sinon
     */
    public function is_connected() {
        $site_id = get_option('webful_site_id');
        $api_key = get_option('webful_api_key');

        return !empty($site_id) && !empty($api_key);
    }

    /**
     * Obtenir les informations du site
     *
     * @return array Informations du site
     */
    public function get_site_info() {
        return array(
            'site_id' => get_option('webful_site_id', ''),
            'api_key' => get_option('webful_api_key', ''),
            'site_url' => get_option('webful_site_url', ''),
            'user_email' => get_option('webful_user_email', ''),
            'tracking_enabled' => get_option('webful_tracking_enabled', '1'),
            'check_quota' => get_option('webful_check_quota', '1'),
            'exclude_admins' => get_option('webful_exclude_admins', '1'),
            'excluded_pages' => get_option('webful_excluded_pages', '')
        );
    }

    /**
     * Déconnecter le site de WEBFUL
     */
    public function disconnect() {
        delete_option('webful_site_id');
        delete_option('webful_api_key');
        delete_option('webful_site_url');
        delete_option('webful_user_email');
    }

    /**
     * Tester la connexion à l'API WEBFUL
     *
     * @return bool True si l'API est accessible, false sinon
     */
    public function test_connection() {
        $response = wp_remote_get($this->api_base . '/check-quota.php', array(
            'timeout' => 10
        ));

        return !is_wp_error($response) && wp_remote_retrieve_response_code($response) !== 0;
    }

    /**
     * Obtenir l'URL de tracking pour le JavaScript
     *
     * @return string URL de l'endpoint de tracking
     */
    public function get_tracking_url() {
        return $this->api_base . '/track.php';
    }

    /**
     * Obtenir l'URL du dashboard WEBFUL
     *
     * @return string URL du dashboard
     */
    public function get_dashboard_url() {
        $site_id = get_option('webful_site_id', '');

        if (empty($site_id)) {
            return str_replace('/api', '/dashboard/', $this->api_base);
        }

        return str_replace('/api', '/dashboard/', $this->api_base) . '?site=' . urlencode($site_id);
    }
}
