<?php
/**
 * Classe de gestion du tracking WEBFUL
 *
 * @package WEBFUL
 * @since 1.0.0
 */

// Empêcher l'accès direct
if (!defined('ABSPATH')) {
    exit;
}

class WEBFUL_Tracker {

    /**
     * Instance de l'API
     */
    private $api;

    /**
     * Constructeur
     */
    public function __construct() {
        $this->api = new WEBFUL_API();
        $this->init_hooks();
    }

    /**
     * Initialiser les hooks WordPress
     */
    private function init_hooks() {
        // Injecter le script de tracking dans le footer
        add_action('wp_footer', array($this, 'inject_tracking_script'), 99);
    }

    /**
     * Injecter le script de tracking dans le footer
     */
    public function inject_tracking_script() {
        // Ne pas tracker si :
        // - Le tracking est désactivé
        // - L'utilisateur est admin (optionnel)
        // - Le site n'est pas configuré
        // - Le quota est atteint (si vérification activée)
        // - La page est exclue

        // Vérifier si le tracking est activé
        if (get_option('webful_tracking_enabled', '1') !== '1') {
            return;
        }

        // Vérifier si le site est configuré
        if (!$this->api->is_connected()) {
            return;
        }

        // Ne pas tracker les administrateurs connectés si l'option est activée
        if (get_option('webful_exclude_admins', '1') === '1' && current_user_can('manage_options')) {
            return;
        }

        // Vérifier si la page actuelle est exclue
        if ($this->is_page_excluded()) {
            return;
        }

        // Vérifier le quota si l'option est activée
        if (get_option('webful_check_quota', '1') === '1') {
            // Utiliser un transient pour mettre en cache le quota (5 minutes)
            $quota_cache = get_transient('webful_quota_check');

            if ($quota_cache === false) {
                $quota = $this->api->check_quota();

                if (is_wp_error($quota)) {
                    // En cas d'erreur, continuer le tracking (fail-safe)
                    set_transient('webful_quota_check', array('quota_ok' => true), 5 * MINUTE_IN_SECONDS);
                } else {
                    $quota_ok = isset($quota['quota_ok']) ? $quota['quota_ok'] : false;
                    set_transient('webful_quota_check', array('quota_ok' => $quota_ok), 5 * MINUTE_IN_SECONDS);

                    if (!$quota_ok) {
                        return; // Quota atteint, ne pas tracker
                    }
                }
            } else {
                if (!$quota_cache['quota_ok']) {
                    return; // Quota atteint (en cache)
                }
            }
        }

        // Obtenir les informations du site
        $site_info = $this->api->get_site_info();
        $site_id = $site_info['site_id'];
        $tracking_url = $this->api->get_tracking_url();

        // URL du script de tracking
        $script_url = WEBFUL_PLUGIN_URL . 'assets/js/webful-tracker.js';

        // Injecter la configuration et le script
        ?>
        <!-- WEBFUL Analytics -->
        <script>
            window.webfulConfig = {
                siteId: <?php echo json_encode($site_id); ?>,
                apiUrl: <?php echo json_encode($tracking_url); ?>
            };
        </script>
        <script src="<?php echo esc_url($script_url); ?>" defer data-cfasync="false"></script>
        <!-- /WEBFUL Analytics -->
        <?php
    }

    /**
     * Vérifier si le quota est OK (méthode publique pour l'admin)
     *
     * @return array|WP_Error Informations sur le quota
     */
    public function get_quota_status() {
        return $this->api->check_quota();
    }

    /**
     * Rafraîchir le cache du quota
     */
    public function refresh_quota_cache() {
        delete_transient('webful_quota_check');
    }

    /**
     * Vérifier si le tracking est actif sur cette page
     *
     * @return bool True si le tracking est actif, false sinon
     */
    public function is_tracking_active() {
        // Le tracking est actif si :
        // - Il est activé dans les options
        // - Le site est configuré
        // - Nous ne sommes pas dans l'admin

        return get_option('webful_tracking_enabled', '1') === '1'
            && $this->api->is_connected()
            && !is_admin();
    }

    /**
     * Vérifier si la page actuelle doit être exclue du tracking
     *
     * @return bool True si la page doit être exclue, false sinon
     */
    private function is_page_excluded() {
        $excluded_pages = get_option('webful_excluded_pages', '');

        // Si aucune page n'est exclue, retourner false
        if (empty($excluded_pages)) {
            return false;
        }

        // Obtenir l'URL actuelle
        $current_url = (is_ssl() ? 'https://' : 'http://') . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];
        $current_path = parse_url($current_url, PHP_URL_PATH);

        // Convertir la liste des pages exclues en tableau (une page par ligne)
        $excluded_list = array_filter(array_map('trim', explode("\n", $excluded_pages)));

        foreach ($excluded_list as $excluded) {
            // Vérifier si l'URL complète correspond
            if (strpos($current_url, $excluded) !== false) {
                return true;
            }

            // Vérifier si le chemin correspond
            if (strpos($current_path, $excluded) !== false) {
                return true;
            }

            // Vérifier si c'est un slug de page/article
            global $post;
            if ($post && $post->post_name === $excluded) {
                return true;
            }
        }

        return false;
    }
}
