<?php
/**
 * Plugin Name: WEBFUL Analytics
 * Plugin URI: https://webful.fr
 * Description: Système d'analyse de trafic ultra-léger et respectueux de la vie privée. Suivez vos statistiques sans ralentir votre site.
 * Version: 1.2.1
 * Author: WEBFUL
 * Author URI: https://webful.fr
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: webful
 * Domain Path: /languages
 * Requires at least: 5.0
 * Requires PHP: 7.4
 */

// Empêcher l'accès direct
if (!defined('ABSPATH')) {
    exit;
}

// Définir les constantes du plugin
define('WEBFUL_VERSION', '1.2.1');
define('WEBFUL_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('WEBFUL_PLUGIN_URL', plugin_dir_url(__FILE__));
define('WEBFUL_PLUGIN_BASENAME', plugin_basename(__FILE__));

// URL de base de l'API WEBFUL
define('WEBFUL_API_BASE', 'https://webful.fr/api');

/**
 * Classe principale du plugin WEBFUL
 */
class WEBFUL_Analytics {

    /**
     * Instance unique du plugin (Singleton)
     */
    private static $instance = null;

    /**
     * Instances des classes
     */
    public $admin;
    public $api;
    public $tracker;

    /**
     * Obtenir l'instance unique du plugin
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructeur privé (Singleton)
     */
    private function __construct() {
        $this->load_dependencies();
        $this->init_hooks();
    }

    /**
     * Charger les dépendances
     */
    private function load_dependencies() {
        require_once WEBFUL_PLUGIN_DIR . 'includes/class-webful-api.php';
        require_once WEBFUL_PLUGIN_DIR . 'includes/class-webful-tracker.php';

        if (is_admin()) {
            require_once WEBFUL_PLUGIN_DIR . 'includes/class-webful-admin.php';
        }
    }

    /**
     * Initialiser les hooks WordPress
     */
    private function init_hooks() {
        // Initialiser les composants
        add_action('plugins_loaded', array($this, 'init_components'));

        // Charger les traductions
        add_action('plugins_loaded', array($this, 'load_textdomain'));

        // Ajouter le lien vers les réglages dans la liste des plugins
        add_filter('plugin_action_links_' . WEBFUL_PLUGIN_BASENAME, array($this, 'add_settings_link'));
    }

    /**
     * Initialiser les composants du plugin
     */
    public function init_components() {
        $this->api = new WEBFUL_API();
        $this->tracker = new WEBFUL_Tracker();

        if (is_admin()) {
            $this->admin = new WEBFUL_Admin();
        }
    }

    /**
     * Charger les traductions
     */
    public function load_textdomain() {
        load_plugin_textdomain('webful', false, dirname(WEBFUL_PLUGIN_BASENAME) . '/languages');
    }

    /**
     * Ajouter un lien "Réglages" dans la liste des plugins
     */
    public function add_settings_link($links) {
        $settings_link = '<a href="' . admin_url('admin.php?page=webful') . '">' . __('Réglages', 'webful') . '</a>';
        array_unshift($links, $settings_link);
        return $links;
    }
}

/**
 * Hook d'activation du plugin
 */
function webful_activate() {
    // Créer les options par défaut
    add_option('webful_site_id', '');
    add_option('webful_api_key', '');
    add_option('webful_tracking_enabled', '1');
    add_option('webful_check_quota', '1');
    add_option('webful_exclude_admins', '1');
    add_option('webful_excluded_pages', '');
    add_option('webful_activated_at', current_time('mysql'));

    // Option de consentement RGPD (par défaut : non consenti)
    add_option('webful_gdpr_consent', '0');
    add_option('webful_gdpr_consent_date', '');

    // Enregistrer la version
    add_option('webful_version', WEBFUL_VERSION);

    // Rediriger vers la page de consentement après activation
    add_option('webful_show_consent_notice', '1');
}
register_activation_hook(__FILE__, 'webful_activate');

/**
 * Hook de désactivation du plugin
 */
function webful_deactivate() {
    // Ne rien faire lors de la désactivation
    // Les options sont conservées au cas où l'utilisateur réactive le plugin
}
register_deactivation_hook(__FILE__, 'webful_deactivate');

/**
 * Hook de désinstallation du plugin
 */
function webful_uninstall() {
    // Supprimer toutes les options
    delete_option('webful_site_id');
    delete_option('webful_api_key');
    delete_option('webful_tracking_enabled');
    delete_option('webful_check_quota');
    delete_option('webful_exclude_admins');
    delete_option('webful_excluded_pages');
    delete_option('webful_activated_at');
    delete_option('webful_version');
    delete_option('webful_user_email');
    delete_option('webful_site_url');
}
register_uninstall_hook(__FILE__, 'webful_uninstall');

/**
 * Initialiser le plugin
 */
function webful_init() {
    return WEBFUL_Analytics::get_instance();
}

// Lancer le plugin
webful_init();
